/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using System;
using System.Collections;
using System.Text;
using Inet.Viewer.Helper;
using System.Drawing;
using System.Text.RegularExpressions;
using Inet.Viewer.Resources;
using System.Globalization;

namespace Inet.Viewer.Data
{


    /// <summary>
    /// This class is the holder of the information needed for a prompt.
    /// Objects of this class represent a prompt with its name, its description,
    /// its default value(s), and other various settings (can be edited, can have
    /// multiple values, etc).<br/>
    /// After the prompts are requested from the user, the prompt values set by the user are
    /// also stored here.
    /// A typical use of this object would be:<br/>
    /// <code><pre>...
    /// PromptData[] myPrompts = new PromptData[3];
    /// for (int i=0; i&lt;3; i++) {
    ///     myPrompts[i] = new PromptData(name[i], description[i], defaultValues[i], defaultDescs[i],
    ///       type[i], discrete[i], range[i], multi[i], changeable[i], descOnly[i], editMask[i], 
    ///       minLength[i], maxLength[i]);
    /// }
    /// MyPromptDialog.execute(myPrompts); //show the prompts, request values from user, etc.
    /// Object[] obj = new Object[3];
    /// for (int i=0; i&lt;3; i++) {
    ///     obj[i] = myPrompts[i].getChosenValue();
    /// }
    /// sendPromptsToServer(obj);
    /// ...
    /// </pre></code>
    /// As can be seen - first the PromptData objects are initialized, then used to request
    /// data from the user (its settings can be retrieved with methods such as getName(), getDescription(), getType(), etc.
    /// <p>The chosen values are of the following type, depending on the type of the prompt:</p>
    /// <table border="1">
    /// <tr><td><b>Prompt Type</b></td><td><b>Java Type</b></td></tr>
    /// <tr><td>Number/Currency</td> <td>Double</td></tr>
    /// <tr><td>Boolean</td> <td>Boolean</td></tr>
    /// <tr><td>Date/DateTime</td> <td>java.util.Date</td></tr>
    /// <tr><td>Time</td> <td>Time</td></tr>
    /// <tr><td>String</td> <td>String</td></tr>
    /// <tr><td>Binary</td> <td>byte[]</td></tr>
    /// <tr><td>Range</td> <td><seealso cref="Range"/></td></tr>
    /// <tr><td>Multiple</td> <td>Vector</td></tr>
    /// </table> 
    /// </summary>
    [Serializable]
    public class PromptData
    {
        private const string PrefixFormula = "formula:";

        /*
         * Number/Currency: Double
         * Boolean: Boolean
         * Date/DateTime: Calendar
         * Time: Time
         * String: String
         * Binary: keine Vernderung
         * 
         * Range: PromptData.Range
         * 
         * Multiple: Vector
         * 
         */


        private const string SubreportPromptPattern = "^#(\\d+)#(.*)";
        //atomar
        /// <summary>
        /// Data type constant - unknown
        /// @since 7.0
        /// </summary>
        public const int Unknown = -1;
        /// <summary>
        /// Data type constant - number
        /// @since 7.0
        /// </summary>
        public const int Number = 6;
        /// <summary>
        /// Data type constant - number
        /// @since 7.0
        /// </summary>
        public const int Currency = 7;
        /// <summary>
        /// Data type constant - boolean
        /// @since 7.0
        /// </summary>
        public const int Boolean = 8;
        /// <summary>
        /// Data type constant - date
        /// @since 7.0
        /// </summary>
        public const int Date = 9;
        /// <summary>
        /// Data type constant - time
        /// @since 7.0
        /// </summary>
        public const int Time = 10;
        /// <summary>
        /// Data type constant - string
        /// @since 7.0
        /// </summary>
        public const int String = 11;
        /// <summary>
        /// Data type constant - datetime
        /// @since 7.0
        /// </summary>
        public const int Datetime = 15;
        /// <summary>
        /// Data type constant - binary
        /// @since 7.0
        /// </summary>
        public const int Binary = 14;


        /// <summary>
        /// Data type bit mask - Range
        /// @since 7.0
        /// </summary>
        public const int Range = 128;
        /// <summary>
        /// Data type bit mask - Array
        /// @since 7.0
        /// </summary>
        public const int Array = 256;

        private string description;
        internal string name;
        private string displayName; // Used in case of subreport prompts, where the "name" must be decoded.
        internal bool rangeAllowed;
        internal bool discreteAllowed;
        internal bool multipleAllowed;
        private bool changeable;
        internal bool descOnly;

        internal int type;
        private ArrayList defaultValues = new ArrayList();
        internal PromptValue values;

        //spter fr Seagate's "Edit Mask", zur Zeit nur "" oder "Password"    
        private string editMask;

        private PromptValue minValue;
        private PromptValue maxValue;

        internal string[] defaultDescs;

        internal bool skipThisPrompt;
        internal bool informix;

        private string cascadingParent;

        /// <summary>
        /// Creates a <code>PromptData</code> object which is to be ignored - this is useful for prompts which
        /// are not used in the report. </summary>
        /// <param name="skipThis">  boolean value whether prompt data object must be ignored
        /// @since 7.0 </param>
        public PromptData(bool skipThis)
        {
            skipThisPrompt = skipThis;
        }

        /// <summary>
        /// Creates a <code>PromptData</code> object with the parameters chosen. </summary>
        /// <param name="name"> Name of the prompt to use as the property name of the prompt </param>
        /// <param name="subreportName"> Name of the sub report this prompt belongs to (or ""
        /// if it is from the main report) </param>
        /// <param name="description"> Description of the prompt </param>
        /// <param name="defaultValues"> DefaultValues if any, that is, list of values to be able to
        ///              choose from. If not a multiple value prompt, the first value is taken
        ///              as the value set by default. </param>
        /// <param name="defaultDescs"> Default value descriptions, if any - indexes must correspond
        ///              to the default value string array. </param>
        /// <param name="type"> Type of the prompt, using the bit flags. For example, a currency range
        ///              array would have the type ARRAY|RANGE|CURRENCY. </param>
        /// <param name="discrete"> Can discrete values be chosen for this prompt? </param>
        /// <param name="range"> Can range values be chosen for this prompt? </param>
        /// <param name="multi"> Can multiple values be chosen for this prompt? </param>
        /// <param name="changeable"> Is it possible enter own values (if not - only default values) </param>
        /// <param name="informixSP"> Is this an Informix Stored Procedure prompt? (default should be false) </param>
        /// <param name="descOnly"> Are only the descriptions of the default values to be shown? </param>
        /// <param name="editMask"> Edit mask - as of yet unimplemented. </param>
        /// <param name="minValue"> Minimum value, or length of value allowed to be entered (if String). </param>
        /// <param name="maxValue"> Maximum value, or length of value allowed to be entered (if String).
        /// @since 7.0 </param>
        public PromptData(string name, string subreportName, string description, string[] defaultValues, string[] defaultDescs, int type, bool discrete, bool range, bool multi, bool changeable, bool descOnly, bool informixSP, string editMask, string minValue, string maxValue)
            : this(name, name, subreportName, description, defaultValues, defaultDescs, type, discrete, range, multi, changeable, descOnly, informixSP, editMask, minValue, maxValue)
        {
        }
        /// <summary>
        /// Creates a <code>PromptData</code> object with the parameters chosen. </summary>
        /// <param name="name"> Name of the prompt to use as the property name of the prompt </param>
        /// <param name="displayName"> Display name of the prompt to use for display only. </param>
        /// <param name="subreportName"> Name of the sub report this prompt belongs to (or ""
        /// if it is from the main report) </param>
        /// <param name="description"> Description of the prompt </param>
        /// <param name="defaultValues"> DefaultValues if any, that is, list of values to be able to
        ///              choose from. If not a multiple value prompt, the first value is taken
        ///              as the value set by default. </param>
        /// <param name="defaultDescs"> Default value descriptions, if any - indexes must correspond
        ///              to the default value string array. </param>
        /// <param name="type"> Type of the prompt, using the bit flags. For example, a currency range
        ///              array would have the type ARRAY|RANGE|CURRENCY. </param>
        /// <param name="discrete"> Can discrete values be chosen for this prompt? </param>
        /// <param name="range"> Can range values be chosen for this prompt? </param>
        /// <param name="multi"> Can multiple values be chosen for this prompt? </param>
        /// <param name="changeable"> Is it possible enter own values (if not - only default values) </param>
        /// <param name="informixSP"> Is this an Informix Stored Procedure prompt? (default should be false) </param>
        /// <param name="descOnly"> Are only the descriptions of the default values to be shown? </param>
        /// <param name="editMask"> Edit mask - as of yet unimplemented. </param>
        /// <param name="minValue"> Minimum value, or length of value allowed to be entered (if String). </param>
        /// <param name="maxValue"> Maximum value, or length of value allowed to be entered (if String).
        /// @since 7.0 </param>
        public PromptData(string name, string displayName, string subreportName, string description, string[] defaultValues, string[] defaultDescs, int type, bool discrete, bool range, bool multi, bool changeable, bool descOnly, bool informixSP, string editMask, string minValue, string maxValue)
            : base()
        {
            this.name = name;
            if (subreportName != null && subreportName.Length > 0)
            {
                Regex subreportPattern = new Regex(SubreportPromptPattern);
                Match m = subreportPattern.Match(name);
                if (m.Success)
                {
                    displayName = (m.Groups[2] + " (" + subreportName + ")");
                }
            }
            this.DisplayName = displayName;
            this.Description = description;
            this.defaultDescs = defaultDescs;
            this.type = type;
            // da ohne Flags!!!!
            this.discreteAllowed = discrete;
            this.rangeAllowed = range;
            this.multipleAllowed = multi;
            this.Changeable = changeable;
            this.descOnly = descOnly;
            this.informix = informixSP;
            this.EditMask = editMask;
            if (minValue != null)
            {
                this.minValue = ExtractValueFromString(minValue, string.Empty);
            }
            if (maxValue != null)
            {
                this.maxValue = ExtractValueFromString(maxValue, string.Empty);
            }

            // * Bin?rprompts werden nicht angezeigt
            // * Wenn "default values only" an ist, und nur ein default Wert angegeben wurde,
            //   UND der Prompt nicht MultiPrompt ist (siehe #13631),
            //   kann man den Default-Wert gleich ?bernehmen.
            if (type == Binary || (!changeable && !multipleAllowed && defaultValues != null && defaultValues.Length == 1))
            {
                skipThisPrompt = true;
            }
            try
            {
                ExtractValuesFromDefault(defaultValues, defaultDescs);
            }
            catch (System.Exception e)
            {
                ViewerUtils.PrintStackTrace(e);
            }
            if (this.DefaultValuesField.Count == 0)
            {
                // #3540, #4051
                this.Changeable = true;
            }
        }

        /// <summary>
        /// Clones a <code>PromptData</code> object and initializes a new one with all values equal to that of the old <code>PromptData</code> object. </summary>
        /// <param name="data"> Prompt to "clone"
        /// @since 7.0 </param>
        public PromptData(PromptData data)
            : base()
        {
            this.skipThisPrompt = data.skipThisPrompt;
            this.name = data.name;
            this.DisplayName = data.DisplayName;
            this.Description = data.Description;
            this.type = data.type;
            this.rangeAllowed = data.RangeAllowed;
            this.discreteAllowed = data.DiscreteAllowed;
            this.multipleAllowed = data.multipleAllowed;
            this.Changeable = data.Changeable;
            this.descOnly = data.descOnly;
            this.EditMask = data.EditMask;
            this.minValue = data.minValue;
            this.maxValue = data.maxValue;
            this.values = data.values;
            if (data.defaultValues != null)
            {
                this.defaultValues = (ArrayList)data.defaultValues.Clone();
            }
            this.defaultDescs = data.defaultDescs;
            this.cascadingParent = data.cascadingParent;
        }

        /// <summary>
        /// goes through the string array, extracting the default values, setting a value in setValues according to them. if no default values are found,
        /// a "default value" according to the prompt type is set in setValues, see <seealso cref="InitializeSetValueAccordingToType()"/>. 
        /// <param name="values"> string array of default values in formula syntax </param>
        /// <param name="descr"> string array of default value descriptions </param>
        /// </summary>
        private void ExtractValuesFromDefault(string[] values, string[] descr)
        {
            if (values.Length == 0)
            {
                InitializeSetValueAccordingToType();
                return;
            }
            ((System.Collections.IList)DefaultValuesField).Clear();

            for (int i = 0; i < values.Length; i++)
            {
                string val = values[i];
                string description = string.Empty;
                if (descr != null && descr.Length >= values.Length && descr[i] != null)
                {
                    description = descr[i];
                }
                try
                {
                    PromptValue pval = ExtractValueFromString(val, description);
                    if (!((System.Collections.ArrayList)DefaultValuesField).Contains(pval))
                    {
                        DefaultValuesField.Add(pval);
                    }
                }
                catch (System.ArgumentException exc)
                {
                    // parse problem
                    ViewerUtils.Error(exc.ToString());
                    ViewerUtils.Error("ignoring this default value: " + val);
                }
            }
            ArrayList clonedValues = (ArrayList)defaultValues.Clone();

            //   if (multipleAllowed) setValues = clonedValues;
            //   else if (clonedValues.size() > 0) setValues = clonedValues.get(0);
            if (MultipleAllowed)
            {
                Values = null;
            }
            else
            {
                if (clonedValues.Count > 0)
                {
                    if (DiscreteAllowed)
                    {
                        for (System.Collections.IEnumerator iterator = clonedValues.GetEnumerator(); iterator.MoveNext(); )
                        {
                            PromptValue val = (PromptValue)iterator.Current;
                            if (val is SinglePromptValue)
                            {
                                Values = val;
                                break;
                            }
                        }
                    }
                    else if (RangeAllowed)
                    {
                        for (System.Collections.IEnumerator iterator = clonedValues.GetEnumerator(); iterator.MoveNext(); )
                        {
                            PromptValue val = (PromptValue)iterator.Current;
                            if (val is RangePromptValue)
                            {
                                Values = val;
                                break;
                            }
                        }

                    }
                }
                else
                {
                    // no values could be found, e.g. because they were not parseable
                    InitializeSetValueAccordingToType();
                }
            }
        }

        /// <summary>
        /// depending on the type of the prompt, setValues is initialized to the corresponding value.
        /// </summary>
        private void InitializeSetValueAccordingToType()
        {
            if (multipleAllowed)
            {
                Values = null;
            }
            else if (DiscreteAllowed)
            {
                Values = new SinglePromptValue(type);
            }
            else if (RangeAllowed)
            {
                if (type == PromptData.Date || type == PromptData.Datetime)
                {
                    Values = null;
                }
                else
                {
                    Values = new RangePromptValue(type);
                }
            }
            else
            {
                Values = new MultiPromptValue(type);
            }
        }

        /// <summary>
        /// Create a PromptValue.
        /// </summary>
        /// <param name="values">a string representation or null</param>
        /// <param name="descr">a description or null</param>
        /// <returns></returns>
        protected internal PromptValue ExtractValueFromString(string values, string descr)
        {
            if (values == null || values.Length == 0 || values.Equals("null", StringComparison.OrdinalIgnoreCase))
            {
                if (DiscreteAllowed)
                {
                    SinglePromptValue value = new SinglePromptValue(null, descr, type);
                    value.OnlyDescription = this.descOnly;
                    return value;
                }
                else if (RangeAllowed)
                {
                    PromptValue value = new RangePromptValue(type);
                    value.OnlyDescription = this.descOnly;
                    return value;
                }
                else
                {
                    PromptValue value = new MultiPromptValue(type);
                    value.OnlyDescription = this.descOnly;
                    return value;
                }
            }
            if (RangeAllowed)
            {
                if (type == String && values.IndexOf("to") != -1)
                {
                    StringTokenizer tokenizer = new StringTokenizer(values, "\"' _", true);
                    bool doubleQuote = false;
                    bool singleQuote = false;
                    bool wasJustUnderscore = false;
                    string value = string.Empty;
                    RangePromptValue r = null;
                    // 
                    string token = string.Empty;
                    while (tokenizer.HasMore())
                    {
                        if (token.Length == 0)
                        {
                            token = tokenizer.Next();
                            continue;
                        }
                        if (!doubleQuote && !singleQuote)
                        {
                            if (token.Equals("\""))
                            {
                                doubleQuote = true;
                                token = string.Empty;
                                continue;
                            }
                            else if (token.Equals("'"))
                            {
                                singleQuote = true;
                                token = string.Empty;
                                continue;
                            }
                            else if (token.Equals("_"))
                            {
                                wasJustUnderscore = true;
                                token = string.Empty;
                                continue;
                            }
                            else if (token.Equals("to"))
                            {
                                token = tokenizer.Next();
                                r = new RangePromptValue(type);
                                r.IncludeLow = !wasJustUnderscore;
                                r.IncludeHigh = !(token.Equals("_"));
                                r.StartValue = new SinglePromptValue(value, string.Empty, type);
                                value = string.Empty;
                                continue;

                            }
                            token = string.Empty;
                            continue;
                        }
                        else
                        {
                            if (doubleQuote && token.Equals("\""))
                            {
                                token = tokenizer.Next();
                                if (!token.Equals("\""))
                                {
                                    doubleQuote = false;
                                    continue;
                                }
                            }
                            else if (singleQuote && token.Equals("'"))
                            {
                                token = tokenizer.Next();
                                if (!token.Equals("'"))
                                {
                                    singleQuote = false;
                                    continue;
                                }
                            }
                        }
                        value += token;
                        token = string.Empty;
                    }
                    if (r != null)
                    {
                        r.EndValue = new SinglePromptValue(value, string.Empty, type);
                        if (descr != null)
                        {
                            r.Description = descr;
                        }
                        return r;
                    }
                }
                else if (type != Binary && type != Boolean)
                {
                    int index = values.IndexOf("to");
                    if (index != -1)
                    {
                        RangePromptValue r = new RangePromptValue(type);
                        if (descr != null)
                        {
                            r.Description = descr;
                        }
                        r.OnlyDescription = this.descOnly;
                        r.IncludeLow = !(values[index - 1] == '_');
                        r.IncludeHigh = !(values[index + 2] == '_');
                        r.StartValue = ((SinglePromptValue)ExtractValueFromString(values.Substring(0, index - (r.IncludeLow ? 0 : 1) - (0)), string.Empty));
                        r.EndValue = ((SinglePromptValue)ExtractValueFromString(values.Substring(index + (r.IncludeHigh ? 2 : 3)), string.Empty));
                        return r;
                    }
                }
            }

            // also isses wohl ein diskreter Wert:

            int[] _params;
            System.DateTime cal;
            SinglePromptValue result = new SinglePromptValue(type);

            switch (type)
            {
                case Number:
                case Currency:
                    if (RangeAllowed)
                    {
                        if (values.Trim().Equals("'<'"))
                        {
                            result = new SinglePromptValue(double.PositiveInfinity, descr, type);
                            break;
                        }
                        else if (values.Trim().Equals("'>'"))
                        {
                            result = new SinglePromptValue(double.NegativeInfinity, descr, type);
                            break;
                        }
                    }

                    result = new SinglePromptValue(double.Parse(values.Trim(), CultureInfo.InvariantCulture), descr, type);
                    break;
                case Boolean:
                    result = new SinglePromptValue(bool.Parse(values), descr, type);
                    break;
                case Date:
                    _params = ExtractDateAndOrTime(values, "DATE", 3);
                    cal = new System.DateTime(_params[0], _params[1], _params[2]);
                    result = new SinglePromptValue(cal, descr, type);

                    break;
                case Time:
                    _params = ExtractDateAndOrTime(values, "TIME", 3);
                    result = new SinglePromptValue(new TimeSpan(_params[0], _params[1], _params[2]), descr, type);

                    break;
                case Datetime:
                    _params = ExtractDateAndOrTime(values, "DATETIME", 6);
                    cal = new System.DateTime(_params[0], _params[1], _params[2], _params[3], _params[4], _params[5]);
                    result = new SinglePromptValue(cal, descr, type);

                    break;
                case String:
                    string stringCopy = values;
                    if (values[0] == '\'' && values[values.Length - 1] == '\'')
                    {
                        stringCopy = stringCopy.Substring(1, stringCopy.Length - 1 - (1));
                        int index = stringCopy.IndexOf("''", 0);
                        while (index != -1)
                        {
                            stringCopy = stringCopy.Substring(0, index - (0)) + stringCopy.Substring(index + 1);
                            index = stringCopy.IndexOf("''", index + 1);
                        }
                    }

                    result = new SinglePromptValue(stringCopy, descr, type);
                    break;
                case Binary:
                    result = new SinglePromptValue(values, descr, type);
                    break;
            }
            result.OnlyDescription = this.descOnly;
            return result;
        }

        private int[] ExtractDateAndOrTime(string values, string tokenToSkip, int howManyParams)
        {
            String[] tokens = values.Split(new string[] { "(", ",", ")", " " }, StringSplitOptions.RemoveEmptyEntries);
            int[] dates = new int[howManyParams];
            int dIndex = 0;
            for (int i = 1; i < tokens.Length && dIndex < howManyParams; i++)
            {
                string token = tokens[i];
                dates[dIndex++] = int.Parse(token);
            }
            return dates;
        }
 
        /// <summary>
        /// Returns the description of this prompt. May be <code>null</code> if no description is set. </summary>
        /// <returns> The description of this prompt. May be <code>null</code> if no description is set.
        /// @since 7.0 </returns>
        public virtual string Description
        {
            get
            {
                return description;
            }
            set
            {
                this.description = value;
            }
        }
        /// <summary>
        /// Returns whether or not this prompt allows range values. </summary>
        /// <returns> Does this prompt allow range values?
        /// @since 7.0 </returns>
        public virtual bool RangeAllowed
        {
            get
            {
                return rangeAllowed;
            }
        }
        /// <summary>
        /// Returns whether or not this prompt allows discrete values. </summary>
        /// <returns> Does this prompt allow discrete values?
        /// @since 7.0 </returns>
        public virtual bool DiscreteAllowed
        {
            get
            {
                return discreteAllowed;
            }
        }
        /// <summary>
        /// Returns whether or not this prompt allows multiple values </summary>
        /// <returns> Does this prompt allow multiple values?
        /// @since 7.0 </returns>
        public virtual bool MultipleAllowed
        {
            get
            {
                return multipleAllowed;
            }
        }
        /// <summary>
        /// Returns the name of this prompt. Note that this name is unique for the report. This name can be null for hidden prompts. </summary>
        /// <returns> The name of this prompt. Note that this name is unique for the report. This name can be null for hidden prompts.
        /// @since 7.0 </returns>
        public virtual string Name
        {
            get
            {
                return name;
            }
        }
        /// <summary>
        /// Returns the type of this prompt (without any included bit flags) - that is,
        /// NUMBER, CURRENCY, BOOLEAN, DATE, TIME, DATETIME, STRING, or BINARY </summary>
        /// <returns> The type of this prompt.</returns>
        /// @since 7.0 
        public virtual int Type
        {
            get
            {
                return type;
            }
        }

        /// <summary>
        /// returns or sets the "set value" for this PromptData instance. To SET a null value, wrap the null in a SinglePromptValue instance.
        /// </summary>
        public virtual PromptValue Values
        {
            get
            {
                return values;
            }
            set
            {
                values = value;
            }
        }

        /// <summary>
        /// Sets the value(s) actually chosen for this prompt. If more than one value
        /// was chosen, use a ArrayList. If values are ranges, use a <seealso cref="Range"/>.
        /// For the values themselves, use:
        /// <ul>
        /// <li>String for STRING</li>
        /// <li>Double for NUMBER and CURRENCY</li>
        /// <li>Boolean for BOOLEAN</li>
        /// <li>Date for DATE and DATETIME</li>
        /// <li>Time for TIME</li>
        /// <li>byte[] for BINARY</li>
        /// </ul> 
        /// </summary>
        public virtual object ChosenValues
        {
            set
            {
                if (value is ArrayList)
                {
                    ArrayList v = (ArrayList)value;
                    IEnumerator it = v.GetEnumerator();
                    ArrayList newV = new ArrayList();
                    while (it.MoveNext())
                    {
                        newV.Add(DecodeValue(it.Current));
                    }
                    values = new MultiPromptValue(newV, Description, type);
                }
                else
                {
                    values = DecodeValue(value);
                }
            }
        }

        // Date -> Calendar fr internes Rechnen
        private PromptValue DecodeValue(object values)
        {
            if (values is DateTime)
            {
                DateTime c = new DateTime();
                c = (DateTime)values;
                values = c;
            }
            if (values is RangePromptValue) // alt
            {
                return ((RangePromptValue)values);
            } // neu
            else if (values is Range)
            {
                return new RangePromptValue((Range)values, type);
            }
            return new SinglePromptValue(values, Description, type);
        }

        /// <summary>
        /// Simply returns the display name of this prompt. 
        /// </summary>
        public override string ToString()
        {
            return DisplayName+"("+Name+")";
        }

        /// <summary>
        /// Returns the icon of this <code>PromptData</code>. </summary>
        /// <returns> Returns the icon of this <code>PromptData</code>. </returns>
        public Bitmap Icon
        {
            get
            {
                switch (type)
                {
                    case Number:
                        return Images.col_b;
                    case Currency:
                        return Images.col_cur;
                    case Boolean:
                        return Images.col_b;
                    case Date:
                        return Images.col_d;
                    case Time:
                        return Images.col_t;
                    case Datetime:
                        return Images.col_dt;
                    case String:
                        return Images.col_s;
                    case Binary:
                        return Images.col_bin;
                    default:
                        return Images.col_unknown;

                }
            }
        }

        /// <summary>
        /// Returns the default values of this prompt.<br/>
        /// Note that Date are given as Calendar object, and DateTime as Timestamp. </summary>
        /// <returns> Default values of this prompt.
        /// @since 7.0 </returns>
        public virtual ArrayList DefaultValues
        {
            get
            {
                ArrayList v = new ArrayList();
                IEnumerator it = DefaultValuesField.GetEnumerator();
                while (it.MoveNext())
                {
                    PromptValue val = (PromptValue)it.Current;
                    v.Add(val);
                }
                return v;
            }
        }

        private static bool StringArraysEqual(string[] s1, string[] s2)
        {
            if (s1 == null)
            {
                s1 = new string[] { };
            }
            if (s2 == null)
            {
                s2 = new string[] { };
            }

            if (s1.Length != s2.Length)
            {
                return false;
            }
            for (int i = 0; i < s1.Length; i++)
            {
                if (!s1[i].Equals(s2[i]))
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// Checks equality with another object - a <code>PromptData</code> object is equal to another
        /// if all settings are the same in both objects. 
        /// <param name="obj">  Object </param>
        /// </summary>
        public override bool Equals(object obj)
        {
            if (!(obj is PromptData))
            {
                return false;
            }
            PromptData p = (PromptData)obj;
            return (p.Changeable == Changeable && p.descOnly == descOnly && p.discreteAllowed == discreteAllowed && p.rangeAllowed == rangeAllowed && p.multipleAllowed == multipleAllowed && p.skipThisPrompt == skipThisPrompt && StringArraysEqual(defaultDescs, p.defaultDescs) && DefaultValuesField.Equals(p.DefaultValuesField) && (p.Description == Description || p.Description.Equals(Description)) && (p.EditMask == EditMask || p.EditMask.Equals(EditMask)) && (p.maxValue == maxValue || p.maxValue.Equals(maxValue)) && (p.minValue == minValue || p.minValue.Equals(minValue)) && (p.name == name || p.name.Equals(name)) && p.type == type);
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public override int GetHashCode()
        {
            StringBuilder valuesAsString = new StringBuilder();
            if (DefaultValuesField != null)
            {
                IEnumerator it = DefaultValuesField.GetEnumerator();
                while (it.MoveNext())
                {
                    string value = it.Current.ToString();
                    valuesAsString.Append(value);
                }
            }
            StringBuilder descsAsString = new StringBuilder();
            if (defaultDescs != null)
            {
                for (int i = 0; i < defaultDescs.Length; i++)
                {
                    string desc = defaultDescs[i];
                    descsAsString.Append(desc);
                }
            }
            string idString = name + type + minValue + maxValue + EditMask + Description + valuesAsString.ToString() + descsAsString.ToString() + multipleAllowed + skipThisPrompt + rangeAllowed + discreteAllowed + descOnly + Changeable;
            return idString.GetHashCode();
        }

        /// <summary>
        /// Checks to see if the given prompt value is within the given limits (min / max limits).
        /// For Strings, the min/max limits are the min/max length of the string, with number and currency fields, they are the
        /// min/max values. </summary>
        /// <param name="value"> PromptValue to check whether it is in the range </param>
        /// <returns> True if value is allowed, false if not </returns>
        internal virtual bool WithinLimits(PromptValue value)
        {
            if (value is RangePromptValue)
            {
                return (WithinLimits(((RangePromptValue)value).StartValue) && WithinLimits(((RangePromptValue)value).EndValue));
            }
            else if (value is MultiPromptValue && value.Value != null)
            {
                bool withinLimits = true;
                ArrayList valueVector = (ArrayList)value.Value;
                for (IEnumerator iter = valueVector.GetEnumerator(); withinLimits && iter.MoveNext(); )
                {
                    PromptValue promptValue = (PromptValue)iter.Current;
                    withinLimits = withinLimits && WithinLimits(promptValue);
                }
                return withinLimits;
            }
            if (value == null || value.Value == null)
            {
                if (multipleAllowed)
                {
                    // Leere Listen sind immer erlaubt
                    return true;
                }
                return (minValue == null && maxValue == null);
            }
            switch (type)
            {
                case Number:
                case Currency:
                    {
                        double v = (double)value.Value;
                        double minV;
                        if (minValue != null)
                        {
                            minV = (double)minValue.Value;
                        }
                        else
                        {
                            minV = double.NegativeInfinity;
                        }
                        double maxV;
                        if (maxValue != null)
                        {
                            maxV = (double)maxValue.Value;
                        }
                        else
                        {
                            maxV = double.PositiveInfinity;
                        }
                        return minV <= v && v <= maxV;
                    }
                case Date:
                case Datetime:
                    {
                        DateTime date = (DateTime)value.Value;
                        DateTime? minV = null;
                        if (minValue != null)
                        {
                            minV = (DateTime)minValue.Value;
                        }
                        DateTime? maxV = null;
                        if (maxValue != null)
                        {
                            maxV = (DateTime)maxValue.Value;
                        }
                        return (minV == null || minV <= date) && (maxV == null || maxV >= date);
                    }
                case Time:
                    {
                        TimeSpan time = (TimeSpan)value.Value;
                        TimeSpan? minV = null;
                        if (minValue != null)
                        {
                            minV = (TimeSpan)minValue.Value;
                        }
                        TimeSpan? maxV = null;
                        if (maxValue != null)
                        {
                            maxV = (TimeSpan)maxValue.Value;
                        }
                        return (minV == null || minV <= time) && (maxV == null || maxV >= time);
                    }
                case Binary:
                case Boolean:
                    return true;
                case String:
                    {
                        string val = ((string)value.Value);
                        double minV;
                        if (minValue != null)
                        {
                            minV = double.Parse((string)minValue.Value, CultureInfo.InvariantCulture);
                        }
                        else
                        {
                            minV = double.NegativeInfinity;
                        }
                        double maxV;
                        if (maxValue != null)
                        {
                            maxV = double.Parse((string)maxValue.Value, CultureInfo.InvariantCulture);
                        }
                        else
                        {
                            maxV = double.PositiveInfinity;
                        }
                        return (minV <= val.Length && val.Length <= maxV);
                    }
            }
            return true;
        }


        /// <summary>
        /// Gets explanation for the limits  
        /// </summary>
        internal virtual string RangeExplanationMsg
        {
            get
            {
                if (type == String)
                {
                    return "\n" + strings.Prompt_MinString + ": " + PromptValue.GetValueString(minValue) + "\n" + strings.Prompt_MaxString + ": " + PromptValue.GetValueString(maxValue);
                }
                return "\n" + strings.min + ": " + PromptValue.GetValueString(minValue) + "  " + strings.max + ": " + PromptValue.GetValueString(maxValue);
            }
        }

        /// <summary>
        /// Returns the parent to this prompt if this is a cascading prompt. If not, this returns <code>null</code>. 
        /// </summary>
        /// <returns> parent to this prompt, or null if there is none </returns>
        public virtual string CascadingParent
        {
            get
            {
                return cascadingParent;
            }
            set
            {
                if (value != null)
                {
                    // denn auch wenn nur ein Record da ist, soll der Prompt
                    // weiterhin angezeigt werden.
                    skipThisPrompt = false;
                }
                this.cascadingParent = value;
            }
        }


        /// <summary>
        /// Sets the name to be used in the display of the prompt name. 
        /// </summary>
        public virtual string DisplayName
        {
            set
            {
                this.displayName = value;
            }
            get
            {
                return displayName;
            }
        }


        /// <summary>
        /// Sets the edit mask for this prompt - either "" or "password" 
        /// </summary>
        internal virtual string EditMask
        {
            set
            {
                this.editMask = value;
            }
            get
            {
                return editMask;
            }
        }

        /// <summary>
        /// Gets the min value or null if no min value is defined.
        /// </summary>
        internal PromptValue MinValue
        {
            get
            {
                return minValue;
            }
        }

        /// <summary>
        /// sets default values field (used to be package-visible variable) 
        /// </summary>
        internal virtual ArrayList DefaultValuesField
        {
            set
            {
                this.defaultValues = value;
            }
            get
            {
                return defaultValues;
            }
        }



        /// <summary>
        /// whether this prompt can be manually set, normally should be true 
        /// </summary>
        internal virtual bool Changeable
        {
            set
            {
                this.changeable = value;
            }
            get
            {
                return changeable;
            }
        }

        /// <summary>
        /// reads the value from the string, whether multiple or not </summary>
        /// <param name="value"> string as coming directly from the URL property. </param>
        /// <returns> created PromptValue </returns>
        protected internal PromptValue CreateFromURLPropertyValue(string value)
        {
            if (value.Trim().StartsWith(PrefixFormula))
            {
                value = value.Trim().Substring(PrefixFormula.Length);
            }
            if (value.Trim().Equals("[]"))
            {
                return new MultiPromptValue(new ArrayList(), null, type);
            }
            if (value.Trim().StartsWith("["))
            {
                return ExtractMultiPromptValueFromString(value);
            }
            else
            {
                try
                {
                    return ExtractValueFromString(value, null);
                }
                catch (System.Exception e)
                {
                    ViewerUtils.Debug("setting prompt value " + value + " caused a problem:");
                    ViewerUtils.Debug(e.ToString());
                    ViewerUtils.Debug("...defaulting to no set value.");
                }
            }
            return null;

        }
        /// <summary>
        /// takes the string and extracts the various values in the string in array syntax,
        /// then sets its values in the prompt data object. expects syntax like "['a','b','c']" </summary>
        /// <param name="value"> value to extract multiple values from, syntax like "['a','b','c']", may not be null </param>
        /// <returns> values from string </returns>
        /// <seealso cref="MultiPromptValue"/>
        private MultiPromptValue ExtractMultiPromptValueFromString(string value)
        {
            if (value.StartsWith("["))
            {
                value = value.Substring(1);
            }
            ArrayList v = new ArrayList();
            Inet.Viewer.Helper.StringTokenizer t = new Inet.Viewer.Helper.StringTokenizer(value, "[],'\"()", true);
            bool doubleQuote = false;
            bool singleQuote = false;
            bool parentheses = false;
            System.Text.StringBuilder tmpToken = new System.Text.StringBuilder();
            while (t.HasMore())
            {
                string token = t.Next();
                tmpToken.Append(token);
                if (!doubleQuote && !singleQuote)
                {
                    if (token.Equals("\""))
                    {
                        doubleQuote = true;
                        continue;
                    }
                    if (token.Equals("'"))
                    {
                        singleQuote = true;
                        continue;
                    }
                    if (token.Equals("("))
                    {
                        parentheses = true;
                        continue;
                    }
                    if (!parentheses)
                    {
                        if (token.Equals("]") || token.Equals(","))
                        {
                            v.Add(ExtractValueFromString(tmpToken.ToString(0, tmpToken.Length - 1 - (0)), null));
                            tmpToken = new System.Text.StringBuilder();
                            continue;
                        }
                    }
                    else
                    {
                        if (token.Equals(")"))
                        {
                            parentheses = false;
                        }
                        continue;
                    }
                }
                if (doubleQuote && token.Equals("\""))
                {
                    doubleQuote = false;
                    continue;
                }
                if (singleQuote && token.Equals("'"))
                {
                    singleQuote = false;
                    continue;
                }
            }
            return new MultiPromptValue(v, null, type);
        }

        /// <summary>
        /// Gets a boolean flag indicating whether this prompt field is hidden. The field
        /// is hidden when no name is defined or only defaults values are allowed with only
        /// one definded default value.
        /// </summary>
        internal bool Hidden {
            get {
                return name == null || !Changeable && !MultipleAllowed && defaultValues.Count <= 1;
            }
        }
    }
}